#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdio.h>
#include <math.h>
#include <avr/pgmspace.h>
#include <stdlib.h>
#include <string.h>
#include <util/delay.h>
#include "multiUART.h"
#include "keypad.h"



// redefine brackets
#define begin {
#define end }


void initialize(void);
void createMIDImessage(char message, char note, char param);
void readProximity();
void updateChannel(uint8_t id);
void sampleSongs();
void readKeys();
void createMIDImessage(char status, char note, char param);



// -------------------  Setup UARTS ------------------------------------------
FILE uart_io_0 = FDEV_SETUP_STREAM(uart_putchar0, uart_getchar0, _FDEV_SETUP_RW);
FILE uart_io_1 = FDEV_SETUP_STREAM(uart_putchar1, uart_getchar1, _FDEV_SETUP_RW);



//------------------------- Midi Buffer --------------------
#define bufferSize 90
unsigned char MIDI_buffer[bufferSize];		//the buffer
volatile unsigned char buffer_bit = 9;		//for sending individual bits in a byte
volatile unsigned char buffer_empty = 1;	
volatile unsigned char *output_ptr, *write_ptr;		//pointers for ring buffer
volatile unsigned char output_tracker = 0;			//tracks buffer output position
volatile unsigned char write_tracker = 0;			//tracks buffer input position


//-------------------- software timers ------------------------------------

//read keys every 40ms
#define kt 20;
volatile unsigned int keyTime = 20;

//140 bpm 
volatile unsigned int st = 214;
volatile unsigned int sampleTime = 214;




// ---------------------- Notes ------------------------------------------
#define C0  0x0C 
#define Cs0 0x0D  
#define D0  0x0E 
#define Ds0 0x0F
#define E0  0x10
#define F0  0x11
#define Fs0 0x12
#define G0  0x13
#define Gs0 0x14
#define A0  0x15
#define As0 0x16
#define B0  0x17
#define C1  0x18 
#define Cs1 0x19  
#define D1  0x1A 
#define Ds1 0x1B
#define E1  0x1C
#define F1  0x1D
#define Fs1 0x1E
#define G1  0x1F
#define Gs1 0x20
#define A1  0x21
#define As1 0x22
#define B1  0x23
#define C2  0x24
#define Cs2 0x25  
#define D2  0x26 
#define Ds2 0x27
#define E2  0x28
#define F2  0x29
#define Fs2 0x2A
#define G2  0x2B
#define Gs2 0x2C
#define A2  0x2D
#define As2 0x2E
#define B2  0x2F
#define C3  0x30  
#define Cs3 0x31  
#define D3  0x32 
#define Ds3 0x33
#define E3  0x34
#define F3  0x35
#define Fs3 0x36
#define G3  0x37
#define Gs3 0x38
#define A3  0x39
#define As3 0x3A
#define B3  0x3B
#define C4  0x3C  
#define Cs4 0x3D  
#define D4  0x3E
#define Ds4 0x3F
#define E4  0x40
#define F4  0x41
#define Fs4 0x42
#define G4  0x43
#define Gs4 0x44
#define A4  0x45
#define As4 0x46
#define B4  0x47



//------------------------ MIDI parameters ------------------------
#define C1_NOTE_OFF 0x80
#define C2_NOTE_OFF 0x81
#define C3_NOTE_OFF 0x82

#define C1_NOTE_ON 0x90
#define C2_NOTE_ON 0x91
#define C3_NOTE_ON 0x92

#define C1_PATCH_CHANGE 0xC0
#define C2_PATCH_CHANGE 0xC1
#define C3_PATCH_CHANGE 0xC2

//Sustain
#define SUS 0xFF


//Volume
uint8_t volume = 0x7F; //0111 1111 = 128


//Proximity sensors
#define TEMPO_ID 0x48
#define VOLUME_ID 0x49

// ----------------------- Patterns -------------------------------------------

int c1_select = 0; //which pattern
int c1_index = 0;  //index within song

int c2_select = 0;
int c2_index = 0;

int c3_select= 0;
int c3_index = 0; 


// button arrangment
	//  0  3  6  9 
	//  1  4  7  10
    //  2  5  8  11


//Channel 1 patterns
const uint8_t pattern_c1[3][16][3] = {
//pattern c10
{
{F2,A2,C3},
{SUS,SUS,SUS},
{F2,A2,C3},
{0,0,0},
{F2,A2,C3},
{SUS,SUS,SUS},
{SUS,SUS,SUS},
{SUS,SUS,SUS},
{F2,A3,C3},
{SUS,SUS,SUS},
{F2,A2,C3},
{0,0,0},
{F2,A2,C3},
{0,0,0},
{0,0,0},
{0,0,0},

},

//pattern c11
{
{F3,0,0},
{0,0,0},
{0,0,0},
{0,0,0},
{0,A3,0},
{0,0,0},
{0,0,0},
{0,0,0},
{0,C4,0},
{0,0,0},
{0,C4,0},
{0,0,0},
{0,A3,0},
{0,0,0},
{0,0,0},
{0,0,0}
},
//pattern c12
{
{F3,A3,C4},
{SUS,SUS,SUS},
{SUS,SUS,SUS},
{SUS,SUS,SUS},
{A4,C4,E4},
{SUS,SUS,SUS},
{SUS,SUS,SUS},
{SUS,SUS,SUS},
{E4,G4,B4},
{SUS,SUS,SUS},
{SUS,SUS,SUS},
{SUS,SUS,SUS},
{0,0,0},
{0,0,0},
{0,0,0},
{0,0,0}
}

};


//Channel 2 patterns
const uint8_t pattern_c2[3][16] = {
//pattern c20
{C2,0,C2,0,E2,0,0,0,E2,0,E2,0, D2, 0, 0},
//pattern c21
{C2,0,0,0,F2,0,0,0,F2,0,0,0,0,D2,0},
//pattern c22
{C2,0,C2,0,E2,0,C2,0,0,C2,0,0,G2,0,0}
};









const uint8_t pattern_c3[3][16][3] = {
//pattern c30
{
{C2,E2,G2},
{0,0,0},
{C2,E2,G2},
{0,0,0},
{A1,C2,E2},
{0,0,0},
{A1,C2,E2},
{0,0,0},
{E1,G1,B1},
{0,0,0},
{0,0,0},
{0,0,0},
{E1,G1,B1},
{0,0,0},
{0,0,0},
{0,0,0},

},

//pattern c31
{
{C3,G3,A3},
{0,0,0},
{0,0,0},
{0,0,0},
{A3,C4,E4},
{0,0,0},
{0,0,0},
{0,0,0},
{C4,E4,G4},
{0,0,0},
{0,0,0},
{0,0,0},
{C4,E4,G4},
{0,0,0},
{0,0,0},
{0,0,0}

},
//pattern c32
{
{F3,0,C4},
{0,0,0},
{0,A3,0},
{0,0,0},

{F3,0,C4},
{0,0,0},
{0,A3,0},
{0,0,0},

{F3,0,C4},
{0,0,0},
{0,A3,0},
{0,0,0},

{0,0,0},
{0,0,0},
{0,0,0},
{0,0,0}
}

};





// ------------- Channel patches -----------------------------------------------



#define SYNTHDRUM 0x75
#define STEELDRUM 0x72
#define TINKLEBELL 0x70

#define PIANO 0x00
#define SlOWSTRINGS 0x31
#define PICKLED 0x22
#define CLEANGT 0x1b

#define GTHARMONIC 0x19
#define SYNSTR 0x32 

//Channel 1
#define C1_PATCH_LEN 5
char c1_patches[5] = {PIANO, GTHARMONIC, SYNTHDRUM,STEELDRUM,TINKLEBELL };
int c1_patch_index = 0;


//Channel 2
#define C2_PATCH_LEN 4
char c2_patches[4] = {PIANO, SlOWSTRINGS, PICKLED, CLEANGT};
int c2_patch_index = 0;


//Channel 3
#define C3_PATCH_LEN 4
char c3_patches[4] = {PIANO, SYNTHDRUM,STEELDRUM,TINKLEBELL };
int c3_patch_index = 0;






//---------------- USART State Machine Vars -----------------------------------
#define ID_PHASE 0
#define VAL_PHASE 1

//state machine logic
volatile char proximity_phase = ID_PHASE;
volatile char button_phase = ID_PHASE;

//ID of a press
volatile uint8_t r_id_0_temp = 0;
volatile uint8_t r_id_0 = 0;

volatile uint8_t r_id_1_temp = 0;
volatile uint8_t r_id_1 = 0;

//value of press
volatile uint8_t r_val_0 = 0;
volatile uint8_t r_val_1 = 0;


//when polling
volatile uint8_t r0_empty = 1;
volatile uint8_t r1_empty = 1;

//message packet received
volatile uint8_t r0_done  = 0;
volatile uint8_t r1_done  = 0;



//*******************************************************************************************
// Creates software timer for tasks every .125ms
//*******************************************************************************************
ISR (TIMER0_COMPA_vect)
{      
  
  if (keyTime>0)	--keyTime;

  if(sampleTime>0)  --sampleTime;
	
}




//*******************************************************************************************
// Reads Button Presses and updates channel variables
//*******************************************************************************************
ISR (USART0_RX_vect)
begin
		//First Byte
		if(button_phase == ID_PHASE){
			button_phase = VAL_PHASE;
			r_id_0_temp = UDR0;
		}
		else {
			r_id_0 = r_id_0_temp;
			r_val_0 = UDR0;
			r0_done = 1;
			button_phase = ID_PHASE;
		}
		r0_empty = 0;
		UCSR0B &=0x7F; //clear interrupt flag
end


//*******************************************************************************************
//Reads Proximty changes and updates channel variables
//*******************************************************************************************
ISR (USART1_RX_vect)
begin
		//First Byte
		if(proximity_phase == ID_PHASE){
			proximity_phase = VAL_PHASE;
			r_id_1_temp = UDR1;

			
		}
		//Second Byte
		else {
			r_id_1 = r_id_1_temp;
			r_val_1 = UDR1;
			r1_done = 1;
			proximity_phase = ID_PHASE;
			
		
		}
	
		r1_empty = 0;
		UCSR1B &=0x7F; //clear interrupt flag



end


//*******************************************************************************************
//Sends MIDI at 31.25 Kbaud (+/- 1%). or UART
//*******************************************************************************************
ISR (TIMER2_COMPA_vect) 
begin

	//MIDI idles at logic 1
	//send out bits, 9 at a time, starting from end
	if ( !buffer_empty ) {

		//first bit is the start bit, logic 0
		if (buffer_bit == 9) {
			
			PORTC &= 0xFE; //0b11111110,  
			buffer_bit--;
		
		}

		//send data bits, LSB FIRST
		else if (buffer_bit > 0) { 
			
			//0xFE = 0b11111110 
			PORTC = (PORTC & 0xfe) | (0x01 & *output_ptr);
			buffer_bit--;
			//shift the byte to be sent by 1 to make next bit in LSB slot
			*output_ptr = *output_ptr >> 1;
		
		}
		
		else {
			
			PORTC |= 0x01; //raise back to idle
			buffer_bit = 9; //setup next byte

			//if queue is empty set the flag
			if ((output_tracker == (write_tracker - 1)) //output has caught up to input
				 || (output_tracker == 31 && write_tracker == 0) )
		   	{ buffer_empty = 1; }

			//update pointers for next transmission
			if (output_tracker < (bufferSize-1)) {
				//go to next byte in memory
				output_tracker++;
				output_ptr++;
			}
			else {
				//wrap ring back to start
				output_tracker = 0;
				output_ptr = output_ptr - (bufferSize-1);
			}
		}
	}

	
	
	//else buffer is empty
	else { PORTC |= 0x01; } //MIDI idles at logic '1'

end


//*******************************************************************************************
// Selects a new timer value to control tempo
//*******************************************************************************************
void changeTempo(char tempo){
	//70 BPM
	if(tempo < 40){
		st = 428;

	}
	
	//80 bpm
	else if (tempo < 80){
		st = 375;
	}

	//110 bpm
	else if (tempo < 120){
		st = 277;
	}
	
	//120 bpm
	else if (tempo < 160){
		st = 250;
	}

	//128 bpm
	else if (tempo < 200){
		st = 234;
	}

	//140
	else if(tempo < 240){
		st =214;
	
	} 
	//160
	else{
		st = 187;
	}

}


//*******************************************************************************************
//Reads from proximity UART1 and writes midi messages
//*******************************************************************************************
void readProximity(){
	uint8_t id,val;
	
	if(!r1_empty){
		if(r1_done){
			id = r_id_1;
			val = r_val_1;
			r1_done = 0;

			//volume (max 127)
			if (id == VOLUME_ID){
				volume = r_val_1 / 2;
			}
						
			//tempo
			else if (id == TEMPO_ID) {
				changeTempo(r_val_1);
			}
		}
		r1_empty = 1;
		UCSR1B |= 0x80;
	}

}

//*******************************************************************************************
// Updates a channel patch that is playing
//*******************************************************************************************
void changePatch(char channel){
	if(channel == 1){
		if (++c1_patch_index == C1_PATCH_LEN){
			c1_patch_index = 0;
		}
		createMIDImessage(C1_PATCH_CHANGE, c1_patches[c1_patch_index], 255);
	}

	if(channel == 2){
		if (++c2_patch_index == C2_PATCH_LEN){
			c2_patch_index = 0;
		}
		createMIDImessage(C2_PATCH_CHANGE, c2_patches[c2_patch_index], 255);
	}

	if(channel == 3){
		if (++c3_patch_index == C3_PATCH_LEN){
			c3_patch_index = 0;
		}
		createMIDImessage(C3_PATCH_CHANGE, c3_patches[c3_patch_index], 255);
	}




}
//*******************************************************************************************
// Updates the channel pattern that is playing
//*******************************************************************************************
void updateChannel(uint8_t id){
	int i;
	uint8_t note, mod;

	
	mod = id % 4;

	fprintf(&uart_io_0, " changed %d \n \r", id);
	
	if(id < 0 || id > 11){
		return;
	}

	// 0 1 2  3
	// 4 5 6  7
	// 8 9 10 11
	

	//Channel 0 on 0, 4, 8
	if( mod == 0){
		//Turn off previous chords
		for(i = 0; i < 3; i++){
			note = pattern_c1[c1_select][c1_index][i];
			createMIDImessage(C1_NOTE_OFF, note, 0X7F);
		}
		c1_select = id;
	}

	//Channel 1 on 1, 5, 9
	else if( mod == 1){
		note = pattern_c2[c2_select][c2_index];
		createMIDImessage(C2_NOTE_OFF, note, 0X7F);
		c2_select = id - 3;
	}
	//Channel 2 on 2, 6, 10
	else if( mod == 2){

		for(i = 0; i < 3; i++){
			note = pattern_c3[c3_select][c3_index][i];
			createMIDImessage(C3_NOTE_OFF, note, 0X7F);
		}
		c3_select = id - 6;
	}
	//Patch change
	else {
		if(id == 3){
			changePatch(1);
		}
		else if (id == 7){
			changePatch(2);
		}
		else if (id == 11){
			changePatch(3);
		}
	
	}


}

//*******************************************************************************************
//Reads from button UART0 and writes midi messages
//*******************************************************************************************
void readKeyPad(){

	uint8_t id,val;
	


		if(r0_done){
			id = r_id_0;
			val = r_val_0;
		
		
			r0_done = 0;
			updateChannel(id);
		}

	UCSR0B |= 0x80; 
}






//*******************************************************************************************
//reads channels, patches, and generates midis
//*******************************************************************************************
void sampleSongs(){
	uint8_t note, i, off_note;

	//----------- Channel 1 ---------------------------------------------------------

	//Turn Off
	if(c1_index == 15)
	{
	
		for(i = 0; i < 3; i++){
			off_note = pattern_c1[c1_select][c1_index][i];
			note = pattern_c1[c1_select][0][i];
			if(off_note && (note != SUS)){
				createMIDImessage(C1_NOTE_OFF, off_note, 0X7F);
			}
		}
		c1_index = 0; 
	}
	else{
		
		for(i = 0; i < 3; i++){
			off_note = pattern_c1[c1_select][c1_index][i];
			note = pattern_c1[c1_select][c1_index +1][i];
			if((note != SUS)){
				createMIDImessage(C1_NOTE_OFF, off_note, 0X7F);
			}
		}
		c1_index++;
		
	}


	//Turn On
	for(i = 0; i < 3; i++)
	{
			note = pattern_c1[c1_select][c1_index][i];
		
			if(note && (note != SUS)){
				
				createMIDImessage(C1_NOTE_ON, note, volume);
			}

	}

	
	//----------- Channel 2 ---------------------------------------------------------
	//Turn Off
	
	if(c2_index == 15)
	{
		off_note = pattern_c2[c2_select][c2_index];
		note = pattern_c2[c2_select][0];
		if(off_note && (note != SUS)){
			createMIDImessage(C2_NOTE_OFF, off_note, 0X7F);
		}
		c2_index = 0; 
	}
	else{
		
		off_note = pattern_c2[c2_select][c2_index];
		note = pattern_c2[c2_select][c2_index+1];
		if(off_note && (note != SUS)){
			createMIDImessage(C2_NOTE_OFF, off_note, 0X7F);
		}
		c2_index++; 
	}

	
	//Turn On
	note = pattern_c2[c2_select][c2_index];
	if(note && (note != SUS)){
		createMIDImessage(C2_NOTE_ON, note, volume);
	}
	

	
	
	//----------- Channel 3 ---------------------------------------------------------
	//Turn Off
	if(c3_index == 15)
	{
		for(i = 0; i < 3; i++){
			off_note = pattern_c3[c3_select][c3_index][i];
			note = pattern_c3[c3_select][0][i];
			if(off_note && (note != SUS)){
				createMIDImessage(C3_NOTE_OFF, note, 0X7F);
			}
		}
		c3_index = 0; 
	}
	else{
		
		for(i = 0; i < 3; i++){
			
			off_note = pattern_c3[c3_select][c3_index][i];
			note = pattern_c3[c3_select][c3_index+1][i];
			if(off_note && (note != SUS)){
				createMIDImessage(C3_NOTE_OFF, note, 0X7F);
			}
		}
		c3_index++; 
	}
	
	//Turn On
	for(i = 0; i < 3; i++)
	{
			note = pattern_c3[c3_select][c3_index][i];
		
			if(note && (note != SUS)){
				
				createMIDImessage(C3_NOTE_ON, note, volume);
			}

	}
	

	

}


//*******************************************************************************************
//create the 3 byte MIDI message
// param is volume  0-127
//*******************************************************************************************
void createMIDImessage(char status, char note, char param) {

	//put 3 bytes in the buffer and update the write tracker
	//remember to wrap around if the tracker goes above the buffer size
	
	MIDI_buffer[write_tracker] = status;

	write_tracker++;
	write_ptr++;
	if(write_tracker > 31) {
		write_tracker = 0;
		write_ptr = write_ptr - (bufferSize-1);
	}

	MIDI_buffer[write_tracker] = note;

	write_tracker++;
	write_ptr++;
	if(write_tracker > 31) {
		write_tracker = 0;
		write_ptr = write_ptr - (bufferSize-1);
	}

	//some commands are only 2 bytes
	if(param != 255){
		MIDI_buffer[write_tracker] = param; 

		write_tracker++;
		write_ptr++;
		if(write_tracker > 31) {
			write_tracker = 0;
			write_ptr = write_ptr - (bufferSize-1);
		}
	
		//it has stuff written
		buffer_empty = 0;
	}
	
}

//*******************************************************************************************
// Initializations
//*******************************************************************************************
void initialize(void) {
	


	//initialize ring buffer read and write byte
	output_ptr = &MIDI_buffer[0]; 
	write_ptr = &MIDI_buffer[0];



	
	//Set timfer 2 to output serial MIDI
	TCCR2B = 0x02; //set to prescale by 8
	OCR2A = 63; //16MHz divided by 8, 64 ticks will trigger ISR at 31.25kHz
	TCCR2A = (1<<WGM21); //set clear on match
	TIMSK2 = (1<<OCIE2A); //initialize interrupt


	DDRC = 0xff;
	PORTC = 0x01; //initialize pin C0 to output for MIDI bits
	


	//setup UARTs
	uart_init();


	//set up timer 0 for 2 mSec timebase 
  	TIMSK0= (1<<OCIE0A);	//turn on timer 0 cmp match ISR 
    OCR0A = 255;  		//set the compare re to 250 time ticks
	//set prescalar to divide by 64
    TCCR0B= 3; //0b00000011;
	// turn on clear-on-match
    TCCR0A= (1<<WGM01);

	//init patches
	changePatch(1);
	changePatch(2);
	changePatch(3);
	
	sei();	
}

//*******************************************************************************************
// Periodically calls sampling tasks
//*******************************************************************************************
int main(void){

	initialize();


	while(1){

		//2 every ms
		if(keyTime == 0) {
			readKeyPad();
			readProximity();
			keyTime = kt;
		}
		

		//sample based on tempo
		if(sampleTime == 0){
			sampleSongs();
			sampleTime = st;
		} 
		
	}

}
